/***************************************************************************
**
**  This file is part of HVGui.
**
**  HVGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-01-29
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "StationPeakModel.h"

namespace HVGui {

  /*!
    \class StationPeakModel StationPeakModel.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  StationPeakModel::StationPeakModel(const QList<CurveResults *>& results, QObject * parent)
    : AbstractPeakModel(parent)
  {
    TRACE;
    _res=results;
    reset();
  }

  /*!
    Description of destructor still missing
  */
  StationPeakModel::~StationPeakModel()
  {
    TRACE;
    qDeleteAll(_peaks);
  }

  CurveResults * StationPeakModel::results(int index)
  {
    TRACE;
    Peak * p=_peaks.at(index);
    return p->_results;
  }

  /*!
    Builds a reference table with all available peaks
  */
  void StationPeakModel::reset()
  {
    TRACE;
    beginResetModel();
    qDeleteAll(_peaks);
    _peaks.clear();

    for(QList<CurveResults *>::iterator it=_res.begin(); it!=_res.end(); it++) {
      CurveResults * res=*it;
      if(res->windowCount()>0) {
        int nPeaks=res->peakCount();
        for(int ip=0; ip<nPeaks; ip++) {
          Peak * p=new Peak;
          p->_results=res;
          p->_index=ip;
          _peaks.append(p);
        }
      }
    }
    endResetModel();
  }

  int StationPeakModel::rowCount(const QModelIndex & parent) const
  {
    TRACE;
    if(!parent.isValid()) {
      return _peaks.count();
    } else {
      return 0;
    }
  }

  int StationPeakModel::columnCount(const QModelIndex & parent) const
  {
    TRACE;
    if(!parent.isValid()) {
      return 4;
    } else {
      return 0;
    }
  }

  QVariant StationPeakModel::data(const QModelIndex &index, int role) const
  {
    TRACE;
    Peak * p=_peaks.at(index.row());
    switch (role) {
    case Qt::DisplayRole:
      switch(index.column()) {
      case 0:
        return p->_results->name();
      case 1:
        return p->_results->peak(p->_index, CurveResults::PeakFrequency);
      case 2:
        return p->_results->peakMinimum(p->_index);
      case 3:
        return p->_results->peakMaximum(p->_index);
      default: break;
      }
      break;
    default:
      break;
    }
    return QVariant();
  }

  QVariant StationPeakModel::headerData(int section, Qt::Orientation orientation, int role) const
  {
    TRACE;
    if(role!=Qt::DisplayRole) return QVariant();
    if(orientation==Qt::Horizontal) {
      switch(section) {
      case 0: return tr("Station");
      case 1: return tr("Frequency (Hz)");
      case 2: return tr("f/stddev (Hz)");
      case 3: return tr("f*stddev (Hz)");
      default: break;
      }
    } else {
      return section+1;
    }
    return QVariant();
  }

  void StationPeakModel::add(double min, double max)
  {
    TRACE;
    for(QList<CurveResults *>::iterator it=_res.begin(); it!=_res.end(); it++) {
      (*it)->addPeak(min, max);
    }
    reset();
  }

  void StationPeakModel::remove(int index)
  {
    TRACE;
    Peak * p=_peaks.at(index);
    p->_results->removePeak(p->_index);
    reset();
  }

  void StationPeakModel::adjust(int index, double min, double max)
  {
    TRACE;
    Peak * p=_peaks.at(index);
    p->_results->setPeak(p->_index, min, max);
    reset();
  }

  void StationPeakModel::save(QTextStream& s, int index)
  {
    CurveResults * r=results(index);
    Peak * p=_peaks.at(index);
    s << r->name()
        << "\t" << r->peak(p->_index, CurveResults::PeakFrequency)
        << "\t" << r->peakMinimum(p->_index)
        << "\t" << r->peakMaximum(p->_index) << "\n";
  }

} // namespace HVGui
