/***************************************************************************
**
**  This file is part of GeopsySLink.
**
**  GeopsySLink is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsySLink is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-04-24
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include "SeedLinkStation.h"
#include "SeedLinkStream.h"

namespace GeopsySLink {

/*!
  \class SeedLinkStation SeedLinkStation.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const QString SeedLinkStation::xmlSeedLinkStationTag="station";

void SeedLinkStation::clear()
{
  qDeleteAll(*static_cast<QList<SeedLinkStream *> *>(this));
  QList<SeedLinkStream *>::clear();
}

void SeedLinkStation::xml_attributes(XML_ATTRIBUTES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  static const QString keys[]={ "name", "network", "description", "stream_check" };
  attributes.add(keys[0], _name);
  attributes.add(keys[1], _network);
  attributes.add(keys[2], _description);
  attributes.add(keys[3], _streamCheck ? "enabled" : "disabled" );
}

bool SeedLinkStation::xml_setAttributes(XML_SETATTRIBUTES_ARGS)
{
  TRACE;
  Q_UNUSED(context)
  for(XMLRestoreAttributeIterator it=attributes.begin(); it!= attributes.end(); it++ ) {
    const StringSection& att=it.key();
    if(att.size()<3) return false;
    switch (att[2].unicode()) {
    case 'm':
      if(att=="name" ) _name=it.value().toString(); else return false;
      break;
    case 't':
      if(att=="network" ) _network=it.value().toString(); else return false;
      break;
    case 's':
      if(att=="description" ) _description=it.value().toString(); else return false;
      break;
    case 'r':
      if(att=="stream_check" ) _streamCheck=it.value()=="enabled"; else return false;
      break;
    case 'g':
      if(att=="begin_seq" ); else return false;
      break;
    case 'd':
      if(att=="end_seq" ); else return false;
      break;
    default:
      return false;
    }
  }
  return true;
}

void SeedLinkStation::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  for(const_iterator it=begin(); it!=end(); it++ ) {
    XMLSaveAttributes attributes;
    (*it)->xml_attributes(attributes, context);
    (*it)->xml_save(s, context, attributes);
  }
}

XMLMember SeedLinkStation::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="stream" ) {
    SeedLinkStream * stream=new SeedLinkStream(this);
    append(stream);
    return XMLMember(stream);
  }
  else return XMLMember(XMLMember::Unknown);
}

/*!
  Convert a stream specification commonly used with seedlink into a
  regular expression.
*/
QRegExp SeedLinkStation::selectorExpression(const QString& streamSelector)
{
  QString regSelector=streamSelector;
  regSelector.replace(" ", "|");
  regSelector.replace(".", "\\.");
  regSelector.replace(QRegExp("( \\?$|^\\?$|^\\? | \\? )"), "[A-Z]{3}");
  regSelector.replace("?", ".");
  regSelector="("+regSelector+")";
  //App::log(QString("Convert stream selector \"%1\" to \"%2\"").arg(streamSelector).arg(regSelector) << endl;
  return QRegExp(regSelector);
}

/*!
  \a streamSelector is a regular expression to match "seedName.type".
  To convert a stream specification commonly used with seedlink into a
  regular expression use selectorExpresion().
*/
QList<int> SeedLinkStation::indexOf(QRegExp streamSelector) const
{
  QList<int> indexes;
  for(int i=count()-1; i>=0;i-- ) {
    const SeedLinkStream * s=at(i);
    if(streamSelector.indexIn(s->seedName())!=-1 ||
       streamSelector.indexIn(s->location()+s->seedName())!=-1) {
      indexes.append(i);
    }
  }
  return indexes;
}

} // namespace GeopsySLink
