/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-09-13
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "FilterParameterWidget.h"

namespace GeopsyGui {

  /*!

  */
  FilterParameterWidget::FilterParameterWidget(QWidget *parent, Qt::WindowFlags f)
      : QWidget(parent, f)
  {
    TRACE;
    setupUi(this);

    // signals and slots connections
    connect(causalEdit, SIGNAL(toggled(bool)), this, SIGNAL(parametersChanged()));
    connect(filtFromFreq, SIGNAL(valueChanged(double)), this, SIGNAL(parametersChanged()));
    connect(filtToFreq, SIGNAL(valueChanged(double)), this, SIGNAL(parametersChanged()));
    connect(orderEdit, SIGNAL(valueChanged(int)), this, SIGNAL(parametersChanged()));
    connect(windowEdit, SIGNAL(parametersChanged()), this, SIGNAL(parametersChanged()));
    connect(windowWidthEdit, SIGNAL(valueChanged(double)), this, SIGNAL(parametersChanged()));

    setDefaultWindow();
  }

  void FilterParameterWidget::updateAllFields()
  {
    TRACE;
    on_filterType_currentIndexChanged(0);
    on_butterworthButton_toggled(true);
    on_frequencyWindowButton_toggled(true);
    windowEdit->updateAllFields();
  }

  void FilterParameterWidget::on_filterType_activated(int)
  {
    TRACE;
    setDefaultWindow();
  }

  void FilterParameterWidget::on_filterType_currentIndexChanged(int)
  {
    TRACE;
    if(filterType->currentIndex()>1) {
      filtToFreq->show();
      toFreqLabel->show();
      fromFreqLabel->setText("From");
      windowWidthLabel->hide();
      windowWidthEdit->hide();
    } else {
      fromFreqLabel->setText("Value");
      filtToFreq->hide();
      toFreqLabel->hide();
      windowWidthLabel->show();
      windowWidthEdit->show();
    }
    emit parametersChanged();
  }

  void FilterParameterWidget::on_butterworthButton_toggled(bool)
  {
    TRACE;
    bool b=butterworthButton->isChecked();
    causalEdit->setEnabled(b);
    orderEdit->setEnabled(b);
    orderLabel->setEnabled(b);
    emit parametersChanged();
  }

  void FilterParameterWidget::setDefaultWindow()
  {
    TRACE;
    if(filterType->currentIndex()>=2) {  // Band pass/reject
      WindowFunctionParameters param;
      windowEdit->getParameters(param);
      param.setShape(WindowFunctionParameters::Tukey);
      windowEdit->setParameters(param);
      windowWidthLabel->setEnabled(false);
      windowWidthEdit->setEnabled(false);
    } else {
      WindowFunctionParameters param;
      windowEdit->getParameters(param);
      param.setShape(WindowFunctionParameters::Cosine);
      windowEdit->setParameters(param);
      windowWidthLabel->setEnabled(true);
      windowWidthEdit->setEnabled(true);
    }
    windowEdit->updateAllFields();
  }

  void FilterParameterWidget::on_frequencyWindowButton_toggled(bool)
  {
    TRACE;
    bool b=frequencyWindowButton->isChecked();
    if(b && !windowEdit->isEnabled()) {
      setDefaultWindow();
    }
    windowEdit->setEnabled(b);
    emit parametersChanged();
  }

  void FilterParameterWidget::getParameters(FilterParameters& param) const
  {
    TRACE;
    switch (filterType->currentIndex()) {
    case 1:
      param.setBand(FilterParameters::HighPass);
      break;
    case 2:
      param.setBand(FilterParameters::BandPass);
      break;
    case 3:
      param.setBand(FilterParameters::BandReject);
      break;
    default:
      param.setBand(FilterParameters::LowPass);
      break;
    }
    param.setFrequencyRange(filtFromFreq->value(), filtToFreq->value());
    if(butterworthButton->isChecked()) {
      param.setOrder(orderEdit->value());
      param.setCausal(causalEdit->isChecked());
      param.setMethod(FilterParameters::Butterworth);
    } else {
      windowEdit->getParameters(param.frequencyWindow());
      param.setMethod(FilterParameters::FrequencyWindow);
      param.setWidth(windowWidthEdit->value()*0.01);
    }
  }

  void FilterParameterWidget::setParameters(const FilterParameters& param)
  {
    TRACE;
    switch (param.band()) {
    case FilterParameters::LowPass:
      filterType->setCurrentIndex(0);
      break;
    case FilterParameters::HighPass:
      filterType->setCurrentIndex(1);
      break;
    case FilterParameters::BandPass:
      filterType->setCurrentIndex(2);
      break;
    case FilterParameters::BandReject:
      filterType->setCurrentIndex(3);
      break;
    }
    filtFromFreq->setValue(param.minimumFrequency());
    filtToFreq->setValue(param.maximumFrequency());
    switch(param.method()) {
    case FilterParameters::Butterworth:
      orderEdit->setValue(param.order());
      causalEdit->setChecked(param.causal());
      break;
    case FilterParameters::FrequencyWindow:
      windowEdit->setParameters(param.frequencyWindow());
      windowWidthEdit->setValue(param.width()*100.0);
      break;
    }
  }

} // namespace GeopsyGui
