/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-05
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyCore.h>
#include <QGpCoreTools.h>

#include "ArrayTimeWindows.h"
#include "ArrayStationSignals.h"

namespace ArrayCore {

  /*!
    \class ArrayTimeWindows ArrayTimeWindows.h
    \brief Array processing kernel

    Must be initialized with setTimeRangeList().
  */

  /*!
    Description of constructor still missing
  */
  ArrayTimeWindows::ArrayTimeWindows(const ArraySelection * array,
                                     const SparseKeepSignal * keep,
                                     const ArrayParameters * param)
    : _parameters(param), _keep(keep)
  {
    TRACE;
    const StationSignals * s=array->array()->first();
    _samplingFrequency=s->samplingFrequency();
    _samplingPeriod=s->samplingPeriod();
    _events=s->firstValidSignal()->database()->seismicEvents();
    _frequency.setRelativeWidth(_parameters->frequencyBandwidth());
    _list.setVerbosityLevel(7);
    _plan=nullptr;
    _frequencyIndex=-1;
  }

  /*!
    Description of destructor still missing
  */
  ArrayTimeWindows::~ArrayTimeWindows()
  {
    TRACE;
    delete _plan;
  }

  double ArrayTimeWindows::timeRangeLength() const
  {
    TRACE;
    ASSERT(!_list.isEmpty());
    return _list.at(0).lengthSeconds();
  }

  /*!
    Calculate the list of time windows for the current frequency.
  */
  void ArrayTimeWindows::setWindows()
  {
    ASSERT(_keep);
    double f=_frequency.center();
    _list.clear();
    _list.add(f, _parameters->windowing(), *_keep, _keep->timeRange().range(), _events);
    if(!_list.isEmpty()) {
      double oversamplingFactor=_parameters->oversamplingFactor();
      _frequencyFilter.calculate(_frequency,
                                 oversamplingFactor*_list.at(0).lengthSeconds(),
                                 _samplingFrequency);
      _frequencyFilter.sqrt();
      if(_parameters->frequencyBandwidth()==0) {
        delete _plan;
        _plan=new FourierPlan(qRound(_list.at(0).lengthSamples(_samplingFrequency)*oversamplingFactor),
                              _samplingPeriod, f);
      }
    }
  }

  void ArrayTimeWindows::setFrequency(double f)
  {
    _frequency.setCenter(f);
    _list.clear();
  }

  /*!
   Frequency is set from the frequency sampling at index \a iOmega.
  */
  void ArrayTimeWindows::setFrequencyIndex(int iOmega)
  {
    TRACE;
    double f=_parameters->frequencySampling().value(iOmega);
    _frequencyIndex=iOmega;
    setFrequency(f);
  }

  QString ArrayTimeWindows::toString(int index) const
  {
    TRACE;
    QString str;
    int nOmega=_parameters->frequencySampling().count();
    if(_list.isEmpty()) {
      str=tr("%1/%2 %3 Hz")
                     .arg(index+1)
                     .arg(nOmega)
                     .arg(_frequency.center());
    } else {
      str=tr("%1/%2 %3 Hz, %4 windows (%5 s)")
                     .arg(index+1)
                     .arg(nOmega)
                     .arg(_frequency.center())
                     .arg(_list.count())
                     .arg(timeRangeLength());
    }
    return str;
  }

} // namespace ArrayCore
