/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-08-29
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveFKTool.h"
#include "FKTool.h"
#include "ActiveFKSteering.h"

namespace ArrayCore {

  /*!
    \class ActiveFKTool ActiveFKTool.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ActiveFKTool::ActiveFKTool()
    : FKTool()
  {
    TRACE;
    _source=nullptr;
    _grid=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  ActiveFKTool::~ActiveFKTool()
  {
    TRACE;
    if(_source) {
      _source->unlock();
    }
    delete _grid;
  }

  void ActiveFKTool::setSource(const SeismicEvent * e)
  {
    TRACE;
    if(_source) {
      _source->unlock();
    }
    _source=e;
    _source->lock();
  }

  bool ActiveFKTool::setParameters(const AbstractParameters &param)
  {
    TRACE;
    const ActiveFKParameters * fkParam=dynamic_cast<const ActiveFKParameters *>(&param);
    if(!fkParam) {
      qDebug() << "Parameters are not ActiveFKParameters in ActiveFKTool::setParameters.";
      return false;
    }
    setGrid(*fkParam);
    return FKTool::setParameters(param);
  }

  void ActiveFKTool::setGrid(const ActiveFKParameters& param)
  {
    TRACE;
    delete _grid;
    _grid=new IrregularGrid2D(param.frequencySampling().count(), param.ySampling().count());
    for(int ix=0; ix<param.frequencySampling().count(); ix++) {
      _grid->setX(ix, param.frequencySampling().value(ix));
    }
    if(param.ySampling().isInversed()) {
      for(int iSlow=0; iSlow<param.ySampling().count(); iSlow++) {
        _grid->setY(param.ySampling().count()-iSlow-1, 1.0/param.ySampling().value(iSlow));
      }
    } else {
      for(int iSlow=0; iSlow<param.ySampling().count(); iSlow++) {
        _grid->setY(iSlow, param.ySampling().value(iSlow));
      }
    }
    _grid->init(std::numeric_limits<double>::quiet_NaN());
  }

  /*!
    Called automatically by setParameters if minimum and maximum
    waveNumber are not set.
  */
  bool ActiveFKTool::setWaveNumberGrid()
  {
    TRACE;
    ASSERT(_source);
    ArraySelection arraySel(array());
    arraySel.clear();
    arraySel.select(_source->position(), parameters()->minimumDistance(), parameters()->maximumDistance());
    ActiveFKSteering st;
    st.setSource(_source->position(), arraySel);
    st.setDistances();
    int n=st.count();
    if(n>0) {
      QVector<double> dist(n);
      for(int i=0; i<n; i++) {
        dist[i]=st.distance(i);
      }
      std::sort(dist.begin(), dist.end());
      double dmin=dist.last()-dist.first();
      parameters()->setKmin(2.0*M_PI/dmin);
      for(int i=n-2; i>=0; i--) {
        double d=dist.at(i+1)-dist.at(i);
        if(d>0.0 && d<dmin) {
          dmin=d;
        }
      }
      parameters()->setKmax(2.0*M_PI/dmin);
      return true;
    } else {
      return false;
    }
  }

} // namespace ArrayCore

