/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AbstractFKFunction_H
#define AbstractFKFunction_H

#include <QGpCoreMath.h>

#include "FKPower.h"
#include "FKCache.h"
#include "ArrayStationSignals.h"
#include "GaussianFrequencyBand.h"
#include "FKSteeringOneComponent.h"
#include "ArrayCoreDLLExport.h"

namespace ArrayCore {

  class FKLoopWorker;
  class GaussianFrequencyBand;
  class FKGradientSearch;
  class FKGridSearch;
  class FKCrossSpectrum;
  class FKPeaks;
  class WaveNumberConverter;

  class ARRAYCORE_EXPORT AbstractFKFunction : public AbstractFunction
  {
  public:
    AbstractFKFunction(FKCache * gridCache);
    ~AbstractFKFunction();

    void setMaximumSlowness(double slow) {_squaredSlowMax=slow*slow;}
    double maximumSlowness() const {return sqrt(_squaredSlowMax);}

    void setMinimumSlowness(double slow) {_squaredSlowMin=slow*slow;}
    double minimumSlowness() const {return sqrt(_squaredSlowMin);}

    void setMinimumWavenumber(double k) {_squaredKminArray=k*k;}
    void setMaximumWavenumber(double k) {_squaredKmaxArray=k*k;}

    void setTotalLimits(double squaredOmega);

    void setAzimuthRange(double min, double max);

    //virtual void setGrid(FKGradientSearch * g, double cacheStep, double step, double size);
    virtual void setGrid(FKGridSearch * g, double step, double size);
    virtual bool initGrid(int n);
    virtual void initGrid(const Point& k, int index);

    virtual bool setSource(const QVector<int>& /*blocks*/) {return true;}

    virtual void resetCrossSpectrum()=0;
    virtual void addCrossSpectrum()=0;
    virtual void meanCrossSpectrum(int nBlocks)=0;
    virtual bool invertCrossSpectrum() {return false;}
    virtual bool setFixedEllipticity(double /*frequency*/) {return true;}

    virtual Point2D waveNumber(const Point& k) const {return k;}

    virtual bool hasEllipticity() const {return false;}
    virtual double ellipticity(const Point& k) {Q_UNUSED(k) return 0.0;}
    virtual bool refine(const Point2D& k, double ell,
                        FKPeaks * results,
                        const WaveNumberConverter& conv);

    virtual double noise(const Point& k) const {Q_UNUSED(k) return 0.0;}
    virtual double power(double optimumPower) const {return optimumPower;}

    double maximumK2() const {return _squaredKmaxTotal;}
    double minimumK2() const {return _squaredKminTotal;}

    virtual bool isOnLimit(const Point& k, double frequency) const;

    inline bool isInsideLimits(const Point& k) const;

    virtual const FKSteering * steering(int index) const {return &FKSteeringOneComponent::cache(_gridCache, index);}
    FKSteering * steering() const {return _steering;}

    void setCrossSpectrum(FKCrossSpectrum * s) {_crossSpectrum=s;}

    virtual void setGradient(FKPower& p) const {Q_UNUSED(p)}
    virtual void setHessian(FKPower& p) const {Q_UNUSED(p)}

    virtual void setGradient(const Point& k, FKPower& p) const {Q_UNUSED(k) Q_UNUSED(p)}
    virtual void setHessian(const Point& k, FKPower& p) const {Q_UNUSED(k) Q_UNUSED(p)}

    /*virtual Point gradient(const Point& p) const;
    virtual double concavity(const Point& p) const;
    virtual Point stepDirection(const Point& p) const;*/

    enum Type {Rayleigh, Love, RayleighFixedEll};
    static AbstractFKFunction * create(Type t, ArrayStations::Mode mode,
                                       const FKParameters * param,
                                       FKCache * gridCache);
  protected:
    FKCache * _gridCache;
    FKSteering * _steering;
    FKCrossSpectrum * _crossSpectrum;
  private:
    double _squaredKminArray;
    double _squaredKmaxArray;
    double _squaredSlowMin;
    double _squaredSlowMax;
    double _squaredKminTotal;
    double _squaredKmaxTotal;
    double _minAzimuth, _maxAzimuth;
  };

  inline bool AbstractFKFunction::isInsideLimits(const Point& k) const
  {
    double k2=k.abs2();
    if(k2<minimumK2() || k2>maximumK2()) {
      return false;
    }
    double az=k.azimuth();
    return _minAzimuth<=az && az<_maxAzimuth;
  }

} // namespace ArrayCore

#endif // AbstractFKFunction_H
