/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-11-02
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include "RefractionCurve.h"
#include "RefractionFactory.h"

namespace QGpCoreWave {

const QString RefractionCurve::xmlRefractionCurveTag="RefractionCurve";

/*!
  \class RefractionCurve RefractionCurve.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

void RefractionCurve::clear()
{
  TRACE;
  Curve<FactoryPoint>::clear();
  _log="";
  _name="";
  _source=0.0;
}

/*!
  Set the index in vector \a xSrc of source. Source x must be in \a xSrc.
*/
void RefractionCurve::linkSource(const ::QVector<double>& xSrc)
{
  TRACE;
  _sourceIndex=xSrc.indexOf(_source);
  ASSERT(_sourceIndex>-1);
}

/*!
  Set the index in vector \a xRec of each sample of the curve. Each sample of the curve must have its
  corresponding value in vector \a xRec.
*/
void RefractionCurve::linkReceivers(const ::QVector<double>& xRec)
{
  TRACE;
  int n=count();
  int nx=xRec.count();
  int ix=0;
  for(int i=0; i< n; i++) {
    while(ix<nx && xRec.at(ix) < at(i).x()) ix++;
    ASSERT(at(i).x()==xRec.at(ix));
    (*this)[i].setIndex(ix);
  }
}

/*!
  Calculate the squared misfit of \a values with respect to this curve.

  \a nValues is incremented by the number of valid values where the comparison was possible.
  \a nData is incremented by the number of valid data where the comparison is possible. In
  all cases, \a nData >= \a nValues and for the best matching case, \a nValues==\a nData.

  \a minMisfit2 is the square of the minimum misfit value that can be reached for each sample individualy.
*/
double RefractionCurve::misfit(int& nValues, int& nData, const RefractionFactory& f, MisfitType type, double minMisfit) const
{
  TRACE;
  double res=0;
  int nx=count();
  const RealValue * values=f.source(_sourceIndex);
  for(int i=0;i<nx;i++) {
    const FactoryPoint& p=at(i);
    double pointMisfit=p.misfit(nValues, nData, values[ p.index() ], type, minMisfit);
    if(pointMisfit>0.0) {
      res+=pointMisfit;
    }
  }
  return res;
}

void RefractionCurve::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s,"name",_name);
  writeProperty(s,"log",_log);
  writeProperty(s,"source",_source);
}

void RefractionCurve::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  const_iterator it;
  for(it=begin(); it!=end(); ++it) {
    it->xml_save(s, context);
  }
}

XMLMember RefractionCurve::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="StatPoint") {
    append(FactoryPoint());
    return XMLMember(&last());
  } else if(tag=="name") return XMLMember(0);
  else if(tag=="log") return XMLMember(1);
  else if(tag=="source") return XMLMember(2);
  else return XMLMember(XMLMember::Unknown);
}

bool RefractionCurve::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  switch (memberID) {
  case 0: _name=content.toString(); return true;
  case 1: _log=content.toString(); return true;
  case 2: _source=content.toDouble(); return true;
  default: return false;
  }
}

} // namespace QGpCoreWave
