/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-11-03
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "AutocorrProxy.h"

namespace QGpCoreWave {

/*!
  \class AutocorrProxy AutocorrProxy.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

QString AutocorrProxy::columnName(int col) const
{
  TRACE;
  switch(col) {
    case 0: return tr("Frequency");
    case 1: return tr("Period");
    case 2: return tr("Autocorr ratio");
    case 3: return tr("Stddev");
    case 4: return tr("Weight");
    default: return QString();
  }
}

QString AutocorrProxy::columnUnit(int col) const
{
  TRACE;
  switch(col) {
    case 0: return tr("Hz");
    case 1: return tr("s");
    default: return QString();
  }
}

QVector<int> AutocorrProxy::savedColumns() const
{
  TRACE;
  QVector<int> l;
  l << 0 << 2 << 3 << 4;
  return l;
}

QVariant AutocorrProxy::columnValue(int sample, int col) const
{
  TRACE;
  const FactoryPoint& p=curve()[sample];
  switch(col) {
  case 0: return p.x();
  case 1: return 1.0/p.x();
  case 2: return p.mean();
  case 3: return p.stddev();
  case 4: return p.weight();
  default: return QVariant();
  }
}

void AutocorrProxy::setColumnValue(int sample, int col, const QVariant & value)
{
  TRACE;
  FactoryPoint& p=curve()[sample];
  switch(col) {
  case 0: p.setX(value.toDouble()); break;
  case 1: p.setX(1.0/value.toDouble()); break;
  case 2: p.setMean(value.toDouble()); break;
  case 3: p.setStddev(value.toDouble()); break;
  case 4: p.setWeight(value.toDouble()); break;
  default: break;
  }
}

QStringList AutocorrProxy::columnFileTypes() const
{
  TRACE;
  static QStringList types;
  if(types.isEmpty()) {
    types << tr("Frequency (Hz)")   // 1
          << tr("Period (s)")       // 2
          << tr("Autocorr ratio")   // 3
          << tr("Stddev")           // 4
          << tr("Weight");          // 5
  }
  return types;
}

QVector<int> AutocorrProxy::defaultColumnFileTypes() const
{
  TRACE;
  QVector<int> l;
  l << 1 << 3 << 4 << 5;
  return l;
}

bool AutocorrProxy::parse(ColumnTextIterator& it)
{
  TRACE;
  const ColumnTextParser * parser=it.parser();
  int nColumns=parser->columnCount();
  ModalCurve& c=curve();
  c.clear();
  bool ok=true;
  while(!it.atEnd() && c.isEmpty()) {
    while(!it.atSectionEnd()) {
      FactoryPoint p;
      for(int iCol=0;iCol<nColumns;iCol++) {
        switch(parser->type(iCol)) {
        case 1:
          p.setX(parser->text(it.currentRow(), iCol).toDouble(&ok));
          break;
        case 2:
          p.setX(1.0/parser->text(it.currentRow(), iCol).toDouble(&ok));
          break;
        case 3:
          p.setMean(parser->text(it.currentRow(), iCol).toDouble(&ok));
          break;
        case 4:
          p.setStddev(parser->text(it.currentRow(), iCol).toDouble(&ok));
          break;
        case 5:
          p.setWeight(parser->text(it.currentRow(), iCol).toDouble(&ok));
          break;
       default:
          break;
        }
        if(!ok) {
          parser->errorParsingColumn(iCol, it.currentRow());
          return false;
        }
      }
      c.append(p);
      it.nextRow();
    }
  }
  // For convenience add a default mode
  if(c.modes().isEmpty()) {
    c.addMode(Mode(Mode::Vertical, 0, 0));
  }
  return true;
}

} // namespace QGpCoreWave
