/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-02-27
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AUTOCORRFACTORY_H
#define AUTOCORRFACTORY_H

#include "Dispersion.h"
#include "ModalFactory.h"
#include "AutocorrRing.h"
#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class DispersionFactory;

class QGPCOREWAVE_EXPORT AutocorrFactory : public ModalFactory
{
public:
  AutocorrFactory(const QVector<AutocorrRing> * rings);
  ~AutocorrFactory();

  int verticalModeCount() const {return maxModeCount(_vertical);}
  int horizontalModeCount() const;

  void init();

  bool isVertical() const {return _vertical;}
  bool isRadial() const {return _radial;}
  bool isTransverse() const {return _transverse;}

  int ringCount() const {return _rings->count();}
  ModalStorage& vertical(int ringIndex) const {return *_vertical[ringIndex];}
  ModalStorage& radial(int ringIndex) const {return *_radial[ringIndex];}
  ModalStorage& transverse(int ringIndex) const {return *_transverse[ringIndex];}

  void calculate(DispersionFactory * disp);
  void calculate(double alphaR, DispersionFactory * disp);
  void calculateVertical(DispersionFactory * disp);
  void calculateHorizontal(double alphaR, DispersionFactory * disp);
  void calculateHorizontal(int iOmega, double alphaR, DispersionFactory * disp);
  void setAlpha(const QList<ModalCurve> curves);
  void setHorizontalAutocorr();

  virtual void setMode (const Mode& m);
  virtual const RealValue * mode(const Mode& m) const;
  virtual int storageCount() const {return _rings->count()*3;}
  inline virtual void writeReportHeader(QDataStream& s) const;
  inline virtual ModalStorage * storage(int storageIndex) const;

  double radialAutocorr(double omega, double r1, double r2, double alpha, double cr, double cl);
  double transverseAutocorr(double omega, double r1, double r2, double alpha, double cr, double cl);
protected:
  virtual ModalStorage * newStorage(int nModes) {return new ModalStorage(nModes, x());}
  ModalStorage ** newHorizontalStorage();
  void upgrade(ModalStorage **& s, int ringIndex, int requestedMode);
private:
  int maxModeCount(ModalStorage ** s) const;
  void setInvalid(ModalStorage ** s);

  const QVector<AutocorrRing> * _rings;
  ModalStorage ** _vertical;
  ModalStorage ** _radial;
  ModalStorage ** _transverse;

  int _nVerticalModes;
  int _nHorizontalModes;
  ModalStorage ** _jRR;
  ModalStorage ** _jLR;
  ModalStorage ** _jRT;
  ModalStorage ** _jLT;
  RealStatisticalValue ** _alpha;
};

inline void AutocorrFactory::writeReportHeader(QDataStream& s) const
{
  TRACE;
  s << _rings->count();
}

inline ModalStorage * AutocorrFactory::storage(int storageIndex) const
{
  TRACE;
  int iRing=storageIndex/3;
  int iComponent=storageIndex-iRing*3;
  switch (iComponent) {
  case 1:
    if(_radial) return _radial[iRing]; else return 0;
  case 2:
    if(_transverse) return _transverse[iRing]; else return 0;
  default:
    if(_vertical) return _vertical[iRing]; else return 0;
  }
}

} // namespace QGpCoreWave

#endif // AUTOCORRFACTORY_H
