/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTFUNCTION_H
#define ABSTRACTFUNCTION_H

#include <QtCore>

#include "QGpCoreMathDLLExport.h"
#include "Point.h"

namespace QGpCoreMath {

  class FunctionSearchMaximum;

  class QGPCOREMATH_EXPORT AbstractFunction
  {
  public:
    AbstractFunction() {}
    virtual ~AbstractFunction() {}

    virtual double value(const Point& p) const=0;
    virtual double value(const Point& p, int index) const {Q_UNUSED(index) return value(p);}

    virtual Point gradient(const Point& p) const {Q_UNUSED(p) return Point();}
    virtual double concavity(const Point& p) const {Q_UNUSED(p) return 0.0;}
    virtual Point stepDirection(const Point& p) const {Q_UNUSED(p) return Point();}

    double value2D(Point& p, AxisType a) const;
    double gradientLength2D(const Point& p, AxisType a) const {return vectorLength(gradient(p), a);}
    double gradientDirection2D(const Point& p, AxisType a) const {return vectorDirection(gradient(p), a);}
    double stepLength2D(const Point& p, AxisType a) const {return vectorLength(stepDirection(p), a);}
    double stepDirection2D(const Point& p, AxisType a) const {return vectorDirection(stepDirection(p), a);}
    double step1D(const Point& p, AxisType a) const {return component(stepDirection(p), a);}

    virtual bool initGrid(int n) {Q_UNUSED(n) return false;}
    virtual void initGrid(const Point& p, int index) {Q_UNUSED(p) Q_UNUSED(index)}
    virtual FunctionSearchMaximum * createMaximum(double val, const Point& pos) const;

    virtual double thirdMinimum() const {return 0.0;}
    virtual double thirdMaximum() const {return 0.0;}
    virtual double thirdStep() const {return 1.0;}
  private:
    static double vectorLength(Point v, AxisType a);
    static double vectorDirection(Point v, AxisType a);
    static inline double component(Point v, AxisType a);
  };

  inline double AbstractFunction::component(Point v, AxisType a)
  {
    switch(a) {
    case XAxis:
      break;
    case YAxis:
      return v.y();
    case ZAxis:
      return v.z();
    }
    return v.x();
  }

} // namespace QGpCoreMath

#endif // ABSTRACTFUNCTION2.H
