/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-12-24
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef STATIONSIGNALS_H
#define STATIONSIGNALS_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"
#include "SubSignalPool.h"

namespace GeopsyCore {

class SparseKeepSignal;

class GEOPSYCORE_EXPORT StationSignals
{
  TRANSLATIONS("StationSignals")
public:
  enum Component {UndefinedComponent = 0, VerticalComponent=1, HorizontalComponent=2, AllComponent=3,
                  AnySingleComponent=7, AnyComponent=11};
  Q_DECLARE_FLAGS(Components, Component)

  StationSignals(Components c=UndefinedComponent);
  virtual ~StationSignals();

  void setComponents(Components c);

  Components components() const {return _components;}
  int nComponents() const {return nComponents(_components);}
  static int nComponents(Components c);
  static Signal::Components component(int index, Components c);

  bool addSignal(Signal * sig);
  bool hasAllComponents() const;
  bool contains(Signal * sig) const;
  static void organizeSubPool(SubSignalPool * subPool);

  const SubSignalPool& originals(int iComp) const {return _signals[iComp];}

  DateTime minTime() const;
  DateTime maxTime() const;
  QString unit() const;

  // General information on station deduced from first valid station
  Signal * firstValidSignal() const;
  Signal * lastValidSignal() const;
  inline double samplingPeriod() const;
  inline double samplingFrequency() const;
  inline DateTime startTime() const;
  inline QString name() const;
  inline QString nameComponent() const;
  inline Signal::Components component() const;
  inline const Point& coordinates() const;
  inline const UtmZone& utmZone() const;

  SparseTimeRange timeRange(const TimeRange& r) const;
  void setSampling(SparseKeepSignal * keep) const;
  void setKeep(SparseKeepSignal * keep, const WindowingParameters& param, int stationIndex) const;
protected:
  SubSignalPool * _signals;
private:
  Components _components;
};

inline double StationSignals::samplingPeriod() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->samplingPeriod() : 0.0;
}

inline double StationSignals::samplingFrequency() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->samplingFrequency() : 0.0;
}

inline DateTime StationSignals::startTime() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->startTime() : DateTime();
}

inline QString StationSignals::name() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->name() : QString();
}

inline QString StationSignals::nameComponent() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->nameComponent() : QString();
}

inline Signal::Components StationSignals::component() const
{
  TRACE;
  Signal * sig=firstValidSignal();
  return sig ? sig->component() : Signal::UndefinedComponent;
}

inline const Point& StationSignals::coordinates() const
{
  static Point dummy;
  Signal * sig=firstValidSignal();
  return sig ? sig->receiver() : dummy;
}

inline const UtmZone& StationSignals::utmZone() const
{
  static UtmZone dummy;
  Signal * sig=firstValidSignal();
  return sig ? sig->utmZone() : dummy;
}

} // namespace GeopsyCore

Q_DECLARE_OPERATORS_FOR_FLAGS(GeopsyCore::StationSignals::Components)

#endif // STATIONSIGNALS_H
