/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  DinverDCCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-14
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FORWARD_H
#define FORWARD_H

#include <DinverCore.h>

#include "ParamGroundModel.h"
#include "TargetList.h"

namespace DinverDCCore {

  class DINVERDCCORE_EXPORT Forward : public AbstractForward
  {
    TRANSLATIONS("Forward")
  public:
    Forward();
    ~Forward();

    bool setParamSpace(ParamGroundModel * gm);
    void setTargets(TargetList * tl) {_targets=tl;}

    ParamGroundModel * paramGroundModel() const {return _paramModel;}
    TargetList * targetList() const {return _targets;}

    virtual AbstractForward * clone() const;
    virtual int maximumThreadCount() const {return INT_MAX;}

    inline virtual double misfit(bool& ok);
    inline virtual void writeReport(ReportWriter * report);
    inline virtual void valueChanged(const Parameter * p=nullptr);
    inline virtual bool isOk();
    inline virtual bool isFussyOk(const Parameter * p);
  protected:
    virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
    virtual XMLMember xml_member(XML_MEMBER_ARGS);
    virtual bool xml_polish(XML_POLISH_ARGS);
  private:
    ParamGroundModel * _paramModel;
    TargetList * _targets;
  };

  inline double Forward::misfit(bool& ok)
  {
    TRACE;
    return _targets->misfit(parameterSpace().variableParameterCount(), ok);
  }

  inline void Forward::valueChanged(const Parameter * p)
  {
    if(!p || _targets->isGroundModelParameter(p)) {
      _paramModel->updateFinalProfiles(static_cast<const GroundParameter *>(p));
      _targets->setVp(); // For constant Nu only
    }
  }

  inline bool Forward::isOk()
  {
    return _targets->isPoissonRatioOk();
  }

  inline bool Forward::isFussyOk(const Parameter * p)
  {
    if(_targets->isGroundModelParameter(p)) {
      return _targets->isPoissonRatioOk(static_cast<const GroundParameter *>(p));
    } else {
      return false;
    }
  }

  inline void Forward::writeReport(ReportWriter * report)
  {
    _targets->writeReport(report);
  }

} // namespace DinverDCCore

#endif // FORWARD_H
