/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-05-15
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRadial.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKSteeringTwoComponentRadial.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRadial HRFKDirectRadial.h
    \brief Radial high resolution FK power with direct steering of a 2-component cross spectrum

    Radial high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is made of raw component E and N.
    Projection on the radial direction is included in the steering matrix.
  */

  HRFKDirectRadial::HRFKDirectRadial(FKCache * gridCache)
    : AbstractFKFunction(gridCache)
  {
    delete _steering;
    _steering=new FKSteeringTwoComponentRadial;
    _steering->setArray(gridCache->array());
    static_cast<FKSteeringTwoComponentRadial *>(_steering)->resize();
  }

  void HRFKDirectRadial::resetCrossSpectrum()
  {
    _crossSpectrum->resetRayleigh();
  }

  void HRFKDirectRadial::addCrossSpectrum()
  {
    _crossSpectrum->addHorizontalRayleigh();
  }

  void HRFKDirectRadial::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRayleigh(nBlocks);
  }

  bool HRFKDirectRadial::invertCrossSpectrum()
  {
    return _crossSpectrum->invertRayleigh();
  }

  bool HRFKDirectRadial::initGrid(int n)
  {
    if(_gridCache->isEmpty(FKCache::TwoComponentRadial)) {
      App::log(tr("Caching steering vectors for 2-component radial ... (%1 values)\n").arg(n) );
      _gridCache->resize(FKCache::TwoComponentRadial, n);
      return true;
    } else {
      return false;
    }
  }

  void HRFKDirectRadial::initGrid(const Point& k, int index)
  {
    FKSteeringTwoComponentRadial::init(_gridCache, index , k);
  }

  double HRFKDirectRadial::value(const Point& kell, int index) const
  {
    if(isInsideLimits(kell)) {
      FKPower p(&FKSteeringTwoComponentRadial::cache(_gridCache, index));
      p.setCaponValue(_crossSpectrum->rayleigh());
      return p.value();
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRadial::value(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKSteeringTwoComponentRadial * s=static_cast<FKSteeringTwoComponentRadial *>(_steering);
      s->initValue(k);
      FKPower p(s);
      p.setCaponValue(_crossSpectrum->rayleigh());
      return p.value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

