/***************************************************************************
**
**  This file is part of vslarray.
**
**  vslarray is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  vslarray is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-24
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyCore.h>
#include <QGpCoreTools.h>
#include "Acquisition.h"
#include "vslarrayInstallPath.h"
#include "vslarrayVersion.h"

PACKAGE_INFO("vslarray", VSLARRAY)

ApplicationHelp * help();

int main(int argc, char ** argv)
{
  // Initialize Qt application
  CoreApplication a(argc, argv, help);

  // Options
  QString database, groupName, destDir;
  double delay=20.0;
  double timeScale=1.0;
  // Check geopsy main arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-db") {
        CoreApplication::checkOptionArg(i, argc, argv);
        database=argv[i];
      } else if(arg=="-group") {
        CoreApplication::checkOptionArg(i, argc, argv);
        groupName=argv[i];
      } else if(arg=="-dest") {
        CoreApplication::checkOptionArg(i, argc, argv);
        destDir=argv[i];
      } else if(arg=="-delay") {
        CoreApplication::checkOptionArg(i, argc, argv);
        delay=atof(argv[i]);
      } else if(arg=="-time-scale") {
        CoreApplication::checkOptionArg(i, argc, argv);
        timeScale=atof(argv[i]);
      } else {
        App::log(tr("vslarray: bad option %1, see -help\n").arg(argv[i]) );
        return 2;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }

  if(database.isEmpty()) {
    App::log(tr("vslarray: missing database, see -help\n") );
    return 2;
  }
  if(groupName.isEmpty()) {
    App::log(tr("vslarray: missing group, see -help\n") );
    return 2;
  }
  if(destDir.isEmpty()) {
    App::log(tr("vslarray: missing destination directory, see -help\n") );
    return 2;
  }
  // Load the main geopsy database
  GeopsyCoreEngine gp;
  SignalDatabase * db=gp.defaultDatabase();
  if(!db->open(database)) {
    App::log(tr("vslarray: error opening database %1\n").arg(database) );
    return 2;
  }
  AbstractSignalGroup * group=db->findGroup(groupName);
  if( !group) {
    App::log(tr("vslarray: unknown group of signals (%1)\n").arg(groupName) );
    return 2;
  }

  Acquisition acq(timeScale);
  if(!acq.setSignals(group->subPool())) {
    App::log(tr("vslarray: bad group, does not contain 3 component signals.\n") );
    return 2;
  }
  if(!acq.setDestination(destDir) ) {
    App::log(tr("vslarray: directory %1 does not exist\n").arg(destDir) );
    return 2;
  }
  acq.setDelay(delay);
  acq.start();

  printf("Virtual SeedLink Array\n"
         "Type help to get information.\n");
  while(true) {
    printf("$ ");
    QString rep=File::readLine(true);
    if(rep.startsWith("start")) {
      acq.startStations(rep.mid(6));
    } else if(rep.startsWith("stop")) {
      acq.stopStations(rep.mid(5));
    } else if(rep.startsWith("status")) {
      acq.status();
    } else if(rep.startsWith("quit") || rep.startsWith("exit")) {
      break;
    } else if(rep.startsWith("help")) {
      printf("Help:\n"
             "start <station list>\n"
             "stop <station list>\n"
             "status\n"
             "quit\n\n"
             "Syntax for station list:\n"
             "  SN1[,SN2][,...]\n"
             "  all\n");
    } else if(!rep.isEmpty()) {
      printf("%s: command not found\n",rep.toLatin1().data());
    }
  }

  acq.stop();

  return 0;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS]" );
  h->setComments( "Simulate an array acquisition through a seed link connection. A GROUP from DATABASE is continuously "
                  "copied to DIR by block of DELAY seconds with Mini-seed format. Start time of each channel are "
                  "random. You can interactively start/stop stations. GROUP must contain 3 component signals. "
                  "This software is designed to work with mseedscan_plugin." );
  h->addGroup("vslarray", "vslarray");
  h->addOption( "-db <DATABASE>", "Open database contained in DATABASE" );
  h->addOption( "-group <GROUP>", "Use group of signals named GROUP" );
  h->addOption( "-dest <DIR>", "Destination directory, usually seiscomp/acquisition/indata" );
  h->addOption( "-delay <DELAY>", "Delay between two sent packets" );
  h->addOption( "-time-scale <FACTOR>", "Accelerate (>1) or decelerate (<1) data replay." );
  return h;
}
