/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-10-08
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Symbol.h"

namespace SciFigs {

/*!
  \class Symbol Symbol.h
  \brief A graphic symbol

*/

const QString Symbol::xmlSymbolTag="Symbol";
const Symbol Symbol::null;

Symbol::Symbol(Type type,double size, Pen pen,Brush brush)
{
  _type=type;
  _size=size;
  if(_size<0) _size=0;
  _pen=pen;
  _brush=brush;
}

Symbol::Symbol(const Symbol& s) : XMLClass()
{
  _type=s._type;
  _size=s._size;
  _pen=s._pen;
  _brush=s._brush;
}

Symbol& Symbol::operator=(const Symbol& o)
{
  _type=o._type;
  _size=o._size;
  _pen=o._pen;
  _brush=o._brush;
  return *this;
}

bool Symbol::operator==(const Symbol& o) const
{
  TRACE;
  return _type==o._type
      && _size==o._size
      && _pen==o._pen
      && _brush==o._brush;
}

void Symbol::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "type",symbolName(_type));
  writeProperty(s, "size",_size);
}

void Symbol::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  _pen.xml_save(s, context);
  _brush.xml_save(s, context);
}

XMLMember Symbol::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="type") return XMLMember(0);
  else if(tag=="size") return XMLMember(1);
  else if(tag=="Pen") return XMLMember(&_pen);
  else if(tag=="Brush") return XMLMember(&_brush);
  return XMLMember(XMLMember::Unknown);
}

bool Symbol::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  switch (memberID) {
  case 0:
    _type=symbolValue(content.toString());
    return true;
  case 1:
    _size=content.toDouble();
    return true;
  default:
    return false;
  }
}

Symbol::Type Symbol::symbolValue(QString s)
{
  if(s.count()>=4) {
    switch(s[2].unicode()) {
    case 'o':
      if(s=="Cross") return Cross;
      break;
    case 'u':
      if(s=="Square") return Square;
      break;
    case 'i':
      if(s=="TriangleUp") return TriangleUp;
      else if(s=="TriangleDown") return TriangleDown;
      else if(s=="Triangle") return TriangleDown;    // For compatibility
      break;
    case 'r':
      if(s=="Circle") return Circle;
      break;
    case 's':
      if(s=="Losange") return Losange;
      break;
    case 'a':
      if(s=="Star") return Star;
      break;
    default:
      break;
    }
  }
  return NoSymbol;
}

QString Symbol::symbolName(Type s)
{
  TRACE;
  switch (s) {
  case Cross: return "Cross";
  case Square: return "Square";
  case Circle: return "Circle";
  case TriangleUp: return "TriangleUp";
  case TriangleDown: return "TriangleDown";
  case Losange: return "Losange";
  case Star: return "Star";
  default: return "NoSymbol";
  }
}

void Symbol::paint(QPainter& p, const QPointF& center, double dotpercm) const
{
  TRACE;
  double size=_size*dotpercm*0.1;
  p.save();
  p.setRenderHints(QPainter::Antialiasing, true);
  p.setBrush(_brush.qbrush(dotpercm) );
  p.setPen(_pen.qpen(dotpercm));
  paint(_type, p, center, size, 0.5*size);
  p.restore();
}

/*QTextStream& operator>>(QTextStream& s, Symbol& sym)
{
  bool ok;
  QString str=s.readLine().trimmed();
  sym.setType((Type)str.section(" ",0,0).toInt(&ok));
  if(!ok) goto error;
  sym.setSize(str.section(" ",1,1).toDouble(&ok));
  if(!ok) goto error;
  Pen p;
  s >> p;
  sym.setPen(p);
  str=s.readLine().trimmed();
  int r=str.section(" ",0,0).toInt(&ok);
  if(!ok) goto error;
  int g=str.section(" ",1,1).toInt(&ok);
  if(!ok) goto error;
  int b=str.section(" ",2,2).toInt(&ok);
  if(!ok) goto error;
  _brush.setColor(QColor(r,g,b));
  _brush.setStyle((Qt::BrushStyle)str.section(" ",3,3).toInt(&ok));
  if(!ok) goto error;
  return s;
error:
  Message::wrongTextFormat(s, translate("Symbol", "Restore symbol"));
  return s;
} */

} // namespace SciFigs
