/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-12-01
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Brush.h"
#include "SciFigsGlobal.h"

namespace SciFigs {

const QString Brush::xmlBrushTag="Brush";

void Brush::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "color",color().name());
  writeProperty(s, "style",styleName(style()));
}

XMLMember Brush::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="color") return XMLMember(0);
  else if(tag=="style") return XMLMember(1);
  return XMLMember(XMLMember::Unknown);
}

bool Brush::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  switch (memberID) {
  case 0: {
      QColor col(content.toString());
      setColor(col);
      return true;
    }
  case 1:
    setStyle(styleValue(content.toString()));
    return true;
  default:
    return false;
  }
}

QString Brush::styleName(Qt::BrushStyle s)
{
  TRACE;
  switch (s) {
  case Qt::SolidPattern: return "SolidPattern";
  case Qt::Dense1Pattern: return "Dense1Pattern";
  case Qt::Dense2Pattern: return "Dense2Pattern";
  case Qt::Dense3Pattern: return "Dense3Pattern";
  case Qt::Dense4Pattern: return "Dense4Pattern";
  case Qt::Dense5Pattern: return "Dense5Pattern";
  case Qt::Dense6Pattern: return "Dense6Pattern";
  case Qt::Dense7Pattern: return "Dense7Pattern";
  case Qt::HorPattern: return "HorPattern";
  case Qt::VerPattern: return "VerPattern";
  case Qt::CrossPattern: return "CrossPattern";
  case Qt::BDiagPattern: return "BDiagPattern";
  case Qt::FDiagPattern: return "DDiagPattern";
  case Qt::DiagCrossPattern: return "DiagCrossPattern";
  default: return "NoBrush";
  }
}

Qt::BrushStyle Brush::styleValue(QString s)
{
  if(s=="SolidPattern") return Qt::SolidPattern;
  else if(s=="Dense1Pattern") return Qt::Dense1Pattern;
  else if(s=="Dense2Pattern") return Qt::Dense2Pattern;
  else if(s=="Dense3Pattern") return Qt::Dense3Pattern;
  else if(s=="Dense4Pattern") return Qt::Dense4Pattern;
  else if(s=="Dense5Pattern") return Qt::Dense5Pattern;
  else if(s=="Dense6Pattern") return Qt::Dense6Pattern;
  else if(s=="Dense7Pattern") return Qt::Dense7Pattern;
  else if(s=="HorPattern") return Qt::HorPattern;
  else if(s=="VerPattern") return Qt::VerPattern;
  else if(s=="CrossPattern") return Qt::CrossPattern;
  else if(s=="BDiagPattern") return Qt::BDiagPattern;
  else if(s=="FDiagPattern") return Qt::FDiagPattern;
  else if(s=="DiagCrossPattern") return Qt::DiagCrossPattern;
  else return Qt::NoBrush;
}

/*!
  Return a QBrush. Default patterns are not used. Instead, they are redrawn to be device independent
  They are drawn on a 2.5x2.5 mm square, with 0.2 mm line weight.

  TODO change and test again with Qt 4.4.0

  The resolution indenpendent pattern does not work properly with Qt 4.3.4. A lot of X11 errors are encountered
  when using setImageTexture() and color is ignored even if image has a depth of 1 (to be fixed in Qt 4.4.0,
  bugs 169502 and 181805, respectively).

  Currently, for screen painting (when \a dotpercm==screen resolution) the default patterns are used. For printing to
  other resolutions, the independent texture is used. Hence, for printing, the mask of the texture is not taken into account.
*/
QBrush Brush::qbrush(double dotpercm) const
{
  TRACE;
  if(dotpercm==SciFigsGlobal::screenResolution()) return *this;
  if(dotpercm<40) return *this;
  switch (style()) {
  case Qt::HorPattern:
  case Qt::VerPattern:
  case Qt::CrossPattern:
  case Qt::FDiagPattern:
  case Qt::BDiagPattern:
  case Qt::DiagCrossPattern:
    break;
  default:
    return *this;
  }
  // For selected patterns, redraw them with a constant size (not depending on resolution)
  int s=(int)floor(dotpercm*0.25+0.5);
  int x;
  //QImage image(s,s, QImage::Format_Mono);
  QImage image(s,s, QImage::Format_ARGB32_Premultiplied);
  QPainter p;
  p.begin(&image);
  p.fillRect(0,0,s,s, Qt::white);
  p.setPen(QPen(color(),(int)round(dotpercm*0.02), Qt::SolidLine, Qt::SquareCap, Qt::MiterJoin));
  //p.fillRect(0,0,s,s, Qt::color0);
  //p.setPen(QPen(Qt::color1,(int)round(dotpercm*0.02), Qt::SolidLine, Qt::SquareCap, Qt::MiterJoin));
  switch (style()) {
  case Qt::HorPattern:
    x=s >> 2;
    p.drawLine(0,x,s,x);
    x=(s*3) >> 2;
    p.drawLine(0,x,s,x);
    break;
  case Qt::VerPattern:
    x=s >> 2;
    p.drawLine(x,0, x, s);
    x=(s*3) >> 2;
    p.drawLine(x,0, x, s);
    break;
  case Qt::CrossPattern:
    x=s >> 2;
    p.drawLine(0,x,s,x);
    p.drawLine(x,0, x, s);
    x=(s*3) >> 2;
    p.drawLine(0,x,s,x);
    p.drawLine(x,0, x, s);
    break;
  case Qt::FDiagPattern:
    x=s >> 1;
    p.drawLine(0,x,x,s);
    p.drawLine(x,0,s,x);
    break;
  case Qt::BDiagPattern:
    x=s >> 1;
    p.drawLine(0,x,x,0);
    p.drawLine(x,s,s,x);
    break;
  case Qt::DiagCrossPattern:
    x=s >> 1;
    p.drawLine(0,x,x,s);
    p.drawLine(x,0,s,x);
    p.drawLine(0,x,x,0);
    p.drawLine(x,s,s,x);
    break;
  default:
    break;
  }
  p.end();
  // TODO: maybe use a new constructor in Qt 4.4.0
  //return QBrush(color(), image);
  return QBrush(image);
}

} // namespace SciFigs
