/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-05-20
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "MagnetoTelluricCurve.h"
#include "MagnetoTelluricFactory.h"

namespace QGpCoreWave {

  const QString MagnetoTelluricCurve::xmlMagnetoTelluricCurveTag="MagnetoTelluricCurve";

/*!
  \class MagnetoTelluricCurve MagnetoTelluricCurve.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

  void MagnetoTelluricCurve::clear()
  {
    TRACE;
    Curve<MagnetoTelluricPoint>::clear();
    _log.clear();
    _name.clear();
  }

  /*!
    Set the index in vector \a x of each sample of the curve. Each sample of the curve must have its
    corresponding value in vector \a x.
  */
  void MagnetoTelluricCurve::linkX(const ::QVector<double>& x)
  {
    TRACE;
    int n=count();
    int nx=x.count();
    int ix=0;
    for(int i=0; i< n; i++) {
      while(ix<nx && x.at(ix) < at(i).x()) ix++;
      ASSERT(at(i).x()==x.at(ix));
      (*this)[i].setIndex(ix);
    }
  }

  void MagnetoTelluricCurve::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s,"name",_name);
    writeProperty(s,"log",_log);
  }

  void MagnetoTelluricCurve::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
  {
    TRACE;
    const_iterator it;
    for(it=begin(); it!=end(); ++it) {
      it->xml_save(s, context);
    }
  }

  XMLMember MagnetoTelluricCurve::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag=="ComplexStatisticalPoint") {
      append(MagnetoTelluricPoint());
      return XMLMember(&last());
    } else if(tag=="name") return XMLMember(0);
    else if(tag=="log") return XMLMember(1);
    else return XMLMember(XMLMember::Unknown);
  }

  bool MagnetoTelluricCurve::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    switch (memberID) {
    case 0: _name=content.toString(); return true;
    case 1: _log=content.toString(); return true;
    default: return false;
    }
  }

  /*!
    Calculate the squared misfit of \a values with respect to this curve.

    \a nValues is incremented by the number of valid values where the comparison was possible.
    \a nData is incremented by the number of valid data where the comparison is possible. In
    all cases, \a nData >= \a nValues and for the best matching case, \a nValues==\a nData.

    \a minMisfit is the square of the minimum misfit value that can be reached for each sample individualy.
  */
  double MagnetoTelluricCurve::misfit(int& nValues, int& nData, const MagnetoTelluricFactory& f, MisfitType type, double minMisfit) const
  {
    TRACE;
    double res=0.0;
    int nx=count();
    const ComplexValue * values=f.impedanceXY();
    for(int i=0;i<nx;i++) {
      const MagnetoTelluricPoint& p=at(i);
      double pointMisfit=p.misfit(nValues, nData, values[p.index()], type, minMisfit);
      if(pointMisfit>0.0) {
        res+=pointMisfit;
      }
    }
    return res;
  }

} // namespace QGpCoreWave
