/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-12-18
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef HISTOGRAM2D_H
#define HISTOGRAM2D_H

#include "Scale.h"
#include "IrregularGrid2D.h"
#include "Point2D.h"
#include "StatisticalPoint.h"

namespace QGpCoreMath {

  class GaussDistribution;

  class QGPCOREMATH_EXPORT Histogram2D : public IrregularGrid2D
  {
  public:
    Histogram2D(int nx, int ny);
    Histogram2D(const Histogram2D& o);
    ~Histogram2D();

    class QGPCOREMATH_EXPORT Sample : public Point2D
    {
    public:
      Sample() : Point2D() {_valid=true;}
      Sample(double x, double y) : Point2D(x, y) {_valid=true;}
      Sample(const Point2D& p) : Point2D(p) {_valid=true;}
      Sample(const Sample& s) : Point2D(s) {_valid=s._valid; _line=s._line;}

      void writeData(QTextStream& s) const;
      void writeLine(QTextStream& s) const;
      bool read(const QString& line, int xColumn=0, int yColumn=1, int validColumn=-1);

      bool isValid() const {return _valid;}
      void setValid(bool v) {_valid=v;}

      const QString& line() const {return _line;}
      void setLine(const QString& l) {_line=l;}
    private:
      bool _valid;
      QString _line;
    };

    bool setXSampling(const SamplingParameters& s);
    bool setXSampling(const QVector<double>& x);
    void setXTitle(const QString& t) {_xTitle=t;}
    void setXTitleInversedScale(const QString& t) {_xTitleInversedScale=t;}
    void setXScaleType(Scale::Type s) {_xScaleType=s;}
    bool setYSampling(const SamplingParameters& s);
    void setYTitle(const QString& t) {_yTitle=t;}
    void setYTitleInversedScale(const QString& t) {_yTitleInversedScale=t;}
    void setYScaleType(Scale::Type s) {_yScaleType=s;}

    void setSamples(const QVector<Sample>& s);
    void addSample(const Point2D& s);
    int sampleCount() const {return _samples.count();}
    void countSamples() {countSamples(this);}
    const Sample& sample(int index) const {return _samples.at(index);}

    void setValid(const Rect& limits, bool v);

    SamplingOptions xSampling() const {return _xSampling;}
    const QString& xTitle() const {return _xTitle;}
    const QString& xTitleInversedScale() const {return _xTitleInversedScale;}
    Scale::Type xScaleType() const {return _xScaleType;}
    SamplingOptions ySampling() const {return _ySampling;}
    const QString& yTitle() const {return _yTitle;}
    const QString& yTitleInversedScale() const {return _yTitleInversedScale;}
    Scale::Type yScaleType() const {return _yScaleType;}

    Curve<RealStatisticalPoint> meanCurve() const;
    Curve<RealStatisticalPoint> medianCurve() const;
    Curve<RealStatisticalPoint> modeCurve() const;
    void validateCurve(Curve<RealStatisticalPoint>& curve, double maxMisfit) const;
    double misfit(double x, const GaussDistribution& gd) const;
    Curve<RealStatisticalPoint> pickAll(double maxMisfit) const;

    void filterThreshold(double t);
    void filterRelativeThreshold(AxisType axis, double t);
    void filterCurveOutsideAbsoluteRange(const Curve<RealStatisticalPoint>& curve, double dy);
    void filterCurveOutsideRelativeRange(const Curve<RealStatisticalPoint>& curve, double factor);
    void filterCurveOutsideSigmaRange(const Curve<RealStatisticalPoint>& curve, double factor);
    void filterCurveInsideAbsoluteRange(const Curve<RealStatisticalPoint>& curve, double dy);
    void filterCurveInsideRelativeRange(const Curve<RealStatisticalPoint>& curve, double factor);
    void filterCurveInsideSigmaRange(const Curve<RealStatisticalPoint>& curve, double factor);
    void filterAboveCurve(const Curve<RealStatisticalPoint>& curve);
    void filterBelowCurve(const Curve<RealStatisticalPoint>& curve);
    void filterOutsideCurves(const Curve<RealStatisticalPoint>& curve1,
                             const Curve<RealStatisticalPoint>& curve2);
    void filterInsideCurves(const Curve<RealStatisticalPoint>& curve1,
                            const Curve<RealStatisticalPoint>& curve2);
    void filterMisfit(const Curve<RealStatisticalPoint>& curve, double maxMisfit);

    bool save(const QString &fileName, const QString& headerLine=QString(), bool maxFormat=false) const;
    bool load(const QString &fileName);

    void write(QTextStream& s) const;
    void read(QTextStream& s);
  private:
    void setStatistics(RealStatisticalPoint &p, double average, double variance) const;
    double misfit(double x, IrregularGrid2D& grid, const GaussDistribution& gd) const;
    double misfit(int ix, IrregularGrid2D& grid, const GaussDistribution& gd) const;
    void misfit(double x, IrregularGrid2D& grid, const GaussDistribution& gd, double maxMisfit);
    void countSamples(IrregularGrid2D * grid) const;

    SamplingOptions _xSampling;
    QString _xTitle, _xTitleInversedScale;
    Scale::Type _xScaleType;
    SamplingOptions _ySampling;
    QString _yTitle, _yTitleInversedScale;
    Scale::Type _yScaleType;
    QVector<Sample> _samples;
  };

} // namespace QGpCoreMath

#endif // HISTOGRAM2D_H
