/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-03-09
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "NormalizationParameterWidget.h"

namespace GeopsyGui {

/*!
  \class NormalizationParameterWidget NormalizationParameterWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
NormalizationParameterWidget::NormalizationParameterWidget(QWidget * parent)
    : QWidget(parent)
{
  TRACE;
  setupUi(this);
  frequencyRange->setRangeOnly();
  frequencyRange->setUnit(tr(" Hz"));
  frequencyRange->setPrecision(2);
  frequencyRange->setSingleStep(0.25);
  frequencyRange->setAdmissibleRange(1e-99, std::numeric_limits<double>::infinity());

  // signals and slots connections
  connect(frequencyRange, SIGNAL(parametersChanged()), this, SIGNAL(parametersChanged()));
  connect(surfaceAttenuation, SIGNAL(toggled(bool)), this, SLOT(typeChanged()));
  connect(volumeAttenuation, SIGNAL(toggled(bool)), this, SLOT(typeChanged()));
  connect(maximumAmplitude, SIGNAL(toggled(bool)), this, SLOT(typeChanged()));
  connect(spectrumEnergy, SIGNAL(toggled(bool)), this, SLOT(typeChanged()));
}

void NormalizationParameterWidget::setExternalRange()
{
  TRACE;
  delete frequencyRange;
  frequencyRange=0;
}

void NormalizationParameterWidget::typeChanged()
{
  TRACE;
  if(frequencyRange) {
    frequencyRange->setEnabled(spectrumEnergy->isChecked());
  }
}

void NormalizationParameterWidget::getParameters(NormalizationParameters & param) const
{
  TRACE;
  param.setType(type());
  if(frequencyRange) {
    SamplingParameters sp;
    frequencyRange->getParameters(sp);
    param.setFrequencyRange(sp.minimum(), sp.maximum());
  }
}

void NormalizationParameterWidget::setParameters(const NormalizationParameters & param)
{
  TRACE;
  switch(param.type()) {
  case NormalizationParameters::SurfaceAttenuation:
    surfaceAttenuation->setChecked(true);
    break;
  case NormalizationParameters::VolumeAttenuation:
    volumeAttenuation->setChecked(true);
    break;
  case NormalizationParameters::MaximumAmplitude:
    maximumAmplitude->setChecked(true);
    break;
  case NormalizationParameters::SpectrumEnergy:
    spectrumEnergy->setChecked(true);
    break;
  case NormalizationParameters::None:
    surfaceAttenuation->setChecked(false);
    volumeAttenuation->setChecked(false);
    maximumAmplitude->setChecked(false);
    spectrumEnergy->setChecked(false);
    break;
  }
  if(frequencyRange) {
    SamplingParameters sp;
    sp.setRange(param.minimumFrequency(), param.maximumFrequency());
    frequencyRange->setParameters(sp);
  }
}

NormalizationParameters::Type NormalizationParameterWidget::type() const
{
  if(surfaceAttenuation->isEnabled() && surfaceAttenuation->isChecked()) {
    return NormalizationParameters::SurfaceAttenuation;
  } else if(volumeAttenuation->isEnabled() && volumeAttenuation->isChecked()) {
    return NormalizationParameters::VolumeAttenuation;
  } else if(maximumAmplitude->isEnabled() && maximumAmplitude->isChecked()) {
    return NormalizationParameters::MaximumAmplitude;
  } else if(spectrumEnergy->isEnabled() && spectrumEnergy->isChecked()) {
    return NormalizationParameters::SpectrumEnergy;
  } else {
    return NormalizationParameters::None;
  }
}

} // namespace GeopsyGui
