/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-12-11
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef TIMERANGELIST_H
#define TIMERANGELIST_H

#include <QGpCoreTools.h>
#include "TimeRange.h"
#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class WindowingParameters;
class SparseKeepSignal;
class SeismicEventTable;

class GEOPSYCORE_EXPORT TimeRangeList: private QList<TimeRange *>
{
  TRANSLATIONS("TimeRangeList")
public:
  TimeRangeList() {_verbosityLevel=1;}
  TimeRangeList(const TimeRangeList& o, int firstIndex=0);
  virtual ~TimeRangeList();

  void add(double frequency, const WindowingParameters& param,
           const SparseKeepSignal& keep, const TimeRange& r);
  void add(double frequency, const WindowingParameters& param, const SparseKeepSignal& keep,
           const TimeRange& r, const SeismicEventTable * events);
  void addBlanks(double frequency, const WindowingParameters& param,
                 const SparseKeepSignal& keep, const TimeRange& r);
  inline TimeRange& addOne(const TimeRange& r);
  inline TimeRange& addOne();
  bool add(QTextStream& s, const QString& beginPattern, const QString& endPattern);
  void add(const TimeRangeList& list);
  void remove(const TimeRange& r);
  void remove(int index);
  void inverse(double frequency, const WindowingParameters& param,
               const SparseKeepSignal& keep, const TimeRange& r);
  QString toString() const;
  bool clear();
  int count() const {return QList<TimeRange *>::count();}
  bool isEmpty() const {return QList<TimeRange *>::isEmpty();}

  typedef QList<TimeRange *>::iterator iterator;
  typedef QList<TimeRange *>::const_iterator const_iterator;
  iterator begin() {return QList<TimeRange *>::begin();}
  iterator end() {return QList<TimeRange *>::end();}
  const_iterator begin() const {return QList<TimeRange *>::begin();}
  const_iterator end() const {return QList<TimeRange *>::end();}

  const TimeRange& at(int index) const {return *QList<TimeRange *>::at(index);}
  TimeRange& at(int index) {return *QList<TimeRange *>::operator[](index);}

  // Query information on list
  double longestWindow() const;
  TimeRange timeRange() const;

  void setVerbosityLevel(int v) {_verbosityLevel=v;}
protected:
  virtual TimeRange * newTimeRange(const DateTime& start, const DateTime& end) {return new TimeRange(start, end);}
  virtual TimeRange * newTimeRange(const DateTime& start, double length) {return new TimeRange(start, length);}
  virtual TimeRange * cloneTimeRange(TimeRange * r) {return new TimeRange(*r);}
  virtual void deleteTimeRange(TimeRange * r) {delete r;}
  void internalClear() {QList<TimeRange *>::clear();}
  void append(TimeRange * r) {QList<TimeRange *>::append(r);}
private:
  void blanksToKeep(SparseKeepSignal& keep, double overlap) const;

  int _verbosityLevel;
};

inline TimeRange& TimeRangeList::addOne(const TimeRange& r)
{
  TimeRange * rr=newTimeRange(r.start(), r.end());
  append(rr);
  return *rr;
}

inline TimeRange& TimeRangeList::addOne()
{
  TimeRange * r=newTimeRange(DateTime(), DateTime());
  append(r);
  return *r;
}

} // namespace GeopsyCore

#endif // TIMERANGELIST_H
