/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-10-19
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "MiniSeedRecords.h"
#include "MetaDataFactory.h"

namespace GeopsyCore {

/*!
  \class MiniSeedRecords MiniSeedRecords.h
  \brief Brief description of class still missing

  A compact storage for Mini Seed records.

  The storage is built with add() in MiniSeedTrace::records().

  The order of calls to add() must follow the chronology of the signal samples.
*/

  REGISTER_METADATA(MiniSeedRecords)

  void MiniSeedRecords::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
  {
    TRACE;

    for(QVector<MiniSeedSequence>::const_iterator it=_records.begin(); it!=_records.end(); it++ ) {
      XMLSaveAttributes attributes;
      it->xml_attributes(attributes, context);
      writeProperty(s, MiniSeedSequence::xmlMiniSeedSequenceTag, attributes);
    }
  }

  XMLMember MiniSeedRecords::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag==MiniSeedSequence::xmlMiniSeedSequenceTag) {
      _records.append(MiniSeedSequence());
      return XMLMember(&_records.last());
    } else return XMLMember(this, false); // Kept for compatibility, setProperty() will be called
  }

  /*!
    Kept for compatibility with past releases
  */
  bool MiniSeedRecords::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(memberID)
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    return fromString(0, content.toString());
  }

  /*!
    Kept for compatibility with past releases
  */
  bool MiniSeedRecords::fromString(const QString&, QString string)
  {
    TRACE;
    _records.clear();
    StringSection valSection(string);
    StringSection field;
    const QChar * ptr=0;
    off_t pos;
    quint32 byteLength;
    while(true) {
      field=valSection.nextField(ptr, "," );
      if(!field.isValid()) return false;
      pos=field.toInt();
      field=valSection.nextField(ptr, "," );
      if(!field.isValid()) return false;
      byteLength=field.toInt();
      add(pos, byteLength);
    }
    return true;
  }

  void MiniSeedRecords::add(off_t pos, quint32 byteLength)
  {
    if(_records.isEmpty() ||
       !_records.last().add(pos, byteLength)) {
      _records.resize(_records.size()+1);
      MiniSeedSequence& r=_records.last();
      r.setFirstPos(pos);
      r.setByteLength(byteLength);
    }
  }

  bool MiniSeedRecords::record(int sequenceIndex, int recordIndex, off_t& pos, quint32& byteLength) const
  {
    const MiniSeedSequence& r=_records.at(sequenceIndex);
    if(recordIndex<r.count()) {
      pos=r.pos(recordIndex);
      byteLength=r.byteLength();
      return true;
    } else {
      return false;
    }
  }

} // namespace GeopsyCore
