/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-03-21
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "CustomAsciiFormatsEditor.h"

class CustomFileFormatsModel: public QAbstractListModel
{
public:
  CustomFileFormatsModel(QObject * parent=nullptr);

  void setFormats(const CustomFileFormats * formats);

  virtual int rowCount(const QModelIndex & parent) const;
  virtual QVariant data(const QModelIndex & index, int role=Qt::DisplayRole) const;
  virtual Qt::ItemFlags flags (const QModelIndex & ) const;
private:
  const CustomFileFormats * _formats;
};

CustomFileFormatsModel::CustomFileFormatsModel(QObject * parent)
  : QAbstractListModel(parent)
{
  TRACE;
  _formats=0;
}

void CustomFileFormatsModel::setFormats(const CustomFileFormats * formats)
{
  TRACE;
  beginResetModel();
  _formats=formats;
  endResetModel();
}

int CustomFileFormatsModel::rowCount(const QModelIndex &) const
{
  TRACE;
  return _formats->count();
}

QVariant CustomFileFormatsModel::data(const QModelIndex & index, int role) const
{
  TRACE;
  if(role==Qt::DisplayRole) {
    return _formats->name(index.row());
  } else {
    return QVariant();
  }
}

Qt::ItemFlags CustomFileFormatsModel::flags (const QModelIndex & index) const
{
  if(_formats->format(index.row())->xml_inherits(AsciiSignalFormat::xmlAsciiSignalFormatTag)) {
    return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
  } else {
    return Qt::NoItemFlags;
  }
}

/*!
  \class CustomAsciiFormatsEditor CustomAsciiFormatsEditor.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
CustomAsciiFormatsEditor::CustomAsciiFormatsEditor(QWidget * parent)
    : Dialog(parent)
{
  TRACE;
  setupUi(this);
  CustomFileFormatsModel * model=new CustomFileFormatsModel(this);
  formatList->setModel(model);
  connect(formatList->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)),
          this, SLOT(selectionChanged()));
  _formats=GeopsyCoreEngine::instance()->customFileFormats();
  updateList();
}

/*!
  Description of destructor still missing
*/
CustomAsciiFormatsEditor::~CustomAsciiFormatsEditor()
{
  TRACE;
}

void CustomAsciiFormatsEditor::updateList()
{
  TRACE;
  static_cast<CustomFileFormatsModel *>(formatList->model())->setFormats(_formats);
}

void CustomAsciiFormatsEditor::selectionChanged()
{
  TRACE;
  bool b=formatList->currentIndex().isValid();
  editButton->setEnabled(b);
  exportButton->setEnabled(b);
  removeButton->setEnabled(b);
}

void CustomAsciiFormatsEditor::on_newButton_clicked()
{
  TRACE;
  AsciiSignalFormatEditor * d=new AsciiSignalFormatEditor(this);
  Settings::getWidget(d);
  if(formatList->currentIndex().isValid()) {
    d->setFormat(*static_cast<AsciiSignalFormat *>(_formats->format(formatList->currentIndex().row())));
  } else {
    AsciiSignalFormat f;
    d->setFormat(f);
  }
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    _formats->add(new AsciiSignalFormat(d->format()));
    updateList();
  }
  delete d;
}

void CustomAsciiFormatsEditor::on_editButton_clicked()
{
  TRACE;
  if(!formatList->currentIndex().isValid()) {
    return;
  }
  AsciiSignalFormatEditor * d=new AsciiSignalFormatEditor(this);
  Settings::getWidget(d);
  d->setFormat(*static_cast<AsciiSignalFormat *>(_formats->format(formatList->currentIndex().row())));
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    *static_cast<AsciiSignalFormat *>(_formats->format(formatList->currentIndex().row()))=d->format();
    updateList();
  }
  delete d;
}

void CustomAsciiFormatsEditor::on_removeButton_clicked()
{
  TRACE;
  if(!formatList->currentIndex().isValid()) {
    return;
  }
  _formats->remove(formatList->currentIndex().row());
  updateList();
}

void CustomAsciiFormatsEditor::on_importButton_clicked()
{
  TRACE;
  static QString title=tr("Open ASCII Signal File Formats");
  QStringList fileNames=Message::getOpenFileNames(title, tr("ASCII Signal File Format (*.ascfmt)"));
  if(!fileNames.isEmpty()) {
    foreach(QString fileName, fileNames) {
      AsciiSignalFormat * format=new AsciiSignalFormat;
      XMLErrorReport xmler(XMLErrorReport::Read);
      xmler.setTitle(title);
      xmler.setFileName(fileName);
      XMLHeader hdr(format);
      if(xmler.exec(hdr.xml_restoreFile(fileName))) {
        _formats->add(format);
      } else {
        delete format;
      }
    }
    updateList();
  }
}

void CustomAsciiFormatsEditor::on_exportButton_clicked()
{
  TRACE;
  if(!formatList->currentIndex().isValid()) {
    return;
  }
  AsciiSignalFormat * f=static_cast<AsciiSignalFormat *>(_formats->format(formatList->currentIndex().row()));
  static QString title=tr("Save ASCII Signal File Format");
  QString fileName=Message::getSaveFileName(title, tr("ASCII Signal File Format (*.ascfmt)"),
                                            f->name()+".ascfmt");
  if(!fileName.isEmpty()) {
    XMLErrorReport xmler(XMLErrorReport::Read);
    xmler.setTitle(title);
    xmler.setFileName(fileName);
    XMLHeader hdr(f);
    xmler.exec(hdr.xml_saveFile(fileName));
  }
}
