/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2015-02-26
**  Copyright: 2015-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ExportOptions.h"

namespace SciFigs {

  /*!
    \class ExportOptions ExportOptions.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ExportOptions::ExportOptions()
  {
    // NO TRACE because used in initialization
    _dpi=0;  // No dpi default
    _preserveScales=false;
  }

  /*!
    Description of destructor still missing
  */
  ExportOptions::~ExportOptions()
  {
    TRACE;
  }

  bool ExportOptions::read(int& argc, char ** argv)
  {
    TRACE;
    int i, j=1;
    for(i=1; i<argc; i++) {
      QByteArray arg=argv[i];
      if(arg[0]=='-') {
        if(arg=="-export" || arg=="-e") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _exportFile=argv[i];
        } else if(arg=="-export-graph") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _exportFileGraph=argv[i];
        } else if(arg=="-format" || arg=="-f") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _exportFormat=argv[i];
        } else if(arg=="-dpi") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _dpi=atoi(argv[i]);
          if(_dpi>1200) _dpi=1200;
          else if(_dpi<0) _dpi=0;
        } else if(arg=="-legend" || arg=="-l") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _legendFile=argv[i];
        } else if(arg=="-preserve-scales") {
          _preserveScales=true;
        } else if(arg=="-colormap" || arg=="-cm" || arg=="-p") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _colorMapFile=argv[i];
        } else if(arg=="-makeup" || arg=="-m") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _makeupObjectFiles.append(argv[i]);
        } else if(arg=="-append-layer" || arg=="-add-layer") { // Last test kept for compatibility
          CoreApplication::checkOptionArg(i, argc, argv);
          _appendLayerFiles.append(argv[i]);
        } else if(arg=="-prepend-layer") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _prependLayerFiles.append(argv[i]);
        } else {
          argv[j++]=argv[i];
        }
      } else {
        argv[j++]=argv[i];
      }
    }
    if(j<argc) {
      argv[j]=nullptr;
      argc=j;
    }
    return true;
  }

  void ExportOptions::help(ApplicationHelp * h)
  {
    TRACE;
    h->addOption("-m, -makeup <FILE>", "Modifies the aspect of the object by the format specified in makeup file (.mkup). A "
                                       "graphicObject makeup file is expected. To get a template export it once with the gui "
                                       "from a context menu (not from the general menu bar which export sheet makeup files).\n"
                                       "Several occurences of this option is supported.");
    h->addOption("-e, -export <FILE>", "Exports contents to FILE with format specified by option '-format' and exits");
    h->addOption("-f, -format <FORMAT>", "Specifies format for option '-export'\n"
                                          "  PAGE   Figue page (default)\n"
                                          "         Save sheet contents with an .page format.\n"
                                          "  LAYER  Figue layers\n"
                                          "         Save layers of plots with a .layer format. If -cpp is used and "
                                                   "various plot are created, a numbered extension _xxxx.layer will be added.\n"
                                          "  PDF    PDF document\n"
                                          "  SVG    Scalable Vector Graphics\n"
                                          "  BMP    Windows Bitmap\n"
                                          "  JPG    Joint Photographic Experts Group\n"
                                          "  JPEG   Joint Photographic Experts Group\n"
                                          "  PNG    Portable Network Graphics\n"
                                          "  PPM    Portable Pixmap\n"
                                          "  XBM    X11 Bitmap\n"
                                          "  XPM    X11 Pixmap");
    h->addOption("-export-graph <BASE_NAME>", "Exports each graph to a file with format specified by option '-format' and exits");
    h->addOption("-dpi <DPI>", "Forces resolution to DPI(dot per inch) in export file");
    h->addOption("-l, -legend <FILE>","Legend to set style and color to lines.");
    h->addOption("-cm, -colormap <FILE>","Color map to set colors.");
    h->addOption("-preserve-scales","Preserve scales for x, y and color map (default=automatically adjust).");
    h->addOption("-append-layer <FILE>", "Adds FILE layer on top of the current layer stack. "
                                         "Several occurences can be used to add more than one layer file.");
    h->addOption("-prepend-layer <FILE>", "Adds FILE layer at the bottom of the current layer stack. "
                                          "Several occurences can be used to add more than one layer file.");
  }

  QString ExportOptions::exportFileGraph(int index) const
  {
    TRACE;
    QFileInfo fi(_exportFileGraph);
    QString suffix=fi.completeSuffix();
    if(!suffix.isEmpty()) {
      suffix="."+suffix;
    }
    return fi.baseName()+QString("_%1").arg(index, 5, 10, QChar('0'))+suffix;
  }

} // namespace SciFigs
