/***************************************************************************
**
**  This file is part of HVCore.
**
**  HVCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-05-25
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "RotateResults.h"
#include "HVParameters.h"

namespace HVCore {

  /*!
    \class RotateResults RotateResults.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  RotateResults::RotateResults(AbstractStation * parent)
    : AbstractResults(parent)
  {
    TRACE;
    _grid=nullptr;
    _gridAccess=nullptr;
    _windowCount=0;
  }

  /*!
    Description of destructor still missing
  */
  RotateResults::~RotateResults()
  {
    TRACE;
    delete _grid;
    delete _gridAccess;
  }

  void RotateResults::clear()
  {
    TRACE;
    if(_grid) {
      _grid->init(0.0);
    }
  }

  void RotateResults::setWindows(int winCount, const HVParameters& param)
  {
    TRACE;
    setFrequencySampling(param.sampling());
    _windowCount=winCount;
    double da=Angle::radiansToDegrees(param.rotationStep());
    int nAzimuth=qFloor(180.0/da)+1;
    int ns=frequencySampling().count();

    delete _grid;
    _grid=new IrregularGrid2D(ns, nAzimuth);

    for(int ix=0; ix<ns; ix++) {
      _grid->setX(ix, param.sampling().value(ix));
    }
    for(int iAzimuth=0; iAzimuth<nAzimuth; iAzimuth++) {
      _grid->setY(iAzimuth, iAzimuth*da);
    }
    _grid->init(0.0);
    _gridAccess=new GridParallelWriter(_grid);
    _gridAccess->lockRows();
  }

  void RotateResults::setWindow(int azimuthIndex, double, const Curve<Point2D>& hv)
  {
    TRACE;
    int n=_grid->nx();
    double * curve=new double[n];
    for(int i=0; i<n; i++) {
      curve[i]=log10(hv.at(_grid->x(i)).y());
    }
    double * values=_gridAccess->lockRow(azimuthIndex);
    for(int i=0; i<n; i++) {
      values[i]+=curve[i];
    }
    _gridAccess->unlock(azimuthIndex);
    delete [] curve;
  }

  void RotateResults::setStatistics()
  {
    TRACE;
    _grid->multiplyValues(1.0/_windowCount);
    _grid->pow10();
  }

  /*!
    Used for the max of the color map
  */
  double RotateResults::maximumAmplitude() const
  {
    TRACE;
    return _grid->maximumValue();
  }


  bool RotateResults::save(QString fileName, QString log) const
  {
    TRACE;
    QFile f(fileName);
    if(!f.open(QIODevice::WriteOnly)) {
      App::log(tr("Unable to open file '%1' for writing\n").arg(fileName));
      return false;
    }
    QTextStream s(&f);
    s << *_grid;
    return true;
  }

} // namespace HVCore

