/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-13
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ActiveModels.h"

namespace DinverCore {

  /*!
    \class ActiveModels ActiveModels.h
    \brief Brief description of class still missing

    Active model index remains valid during the whole life of this object.
    Exception is any call to clear() but this is not part of normal inversion
    run. Inversion is always stopped when clear() is called.
  */

  /*!
    Description of constructor still missing
  */
  ActiveModels::ActiveModels(ModelSet * allModels)
      : IncreaseStorage(1024)
  {
    _allModels=allModels;
    _infos=static_cast<ActiveModel *>(allocateVector(sizeof(ActiveModel)));
  }

  /*!
    Description of destructor still missing
  */
  ActiveModels::~ActiveModels()
  {
    TRACE;
    free(_infos);
  }

  void ActiveModels::reallocate()
  {
    _infos=static_cast<ActiveModel *>(reallocateVector((uchar *)_infos, sizeof(ActiveModel)));
  }

  /*!
    Used only when deleting all models from ensemble. Do not call this
    function while an inversion is running.
  */
  void ActiveModels::clear()
  {
    free(_infos);
    IncreaseStorage::clear();
    _infos=static_cast<ActiveModel *>(allocateVector(sizeof(ActiveModel)));
  }

  void ActiveModels::print(int activeIndex)
  {
    printf("Active index %5i: ", activeIndex);
    _allModels->print(_infos[ activeIndex ].modelIndex());
  }

  void ActiveModels::addNavigatorHits(const QVector<int>& activeIndexes)
  {
    for(QVector<int>::const_iterator it=activeIndexes.begin(); it!=activeIndexes.end(); it++ ) {
      at(*it).addNavigatorHit();
    }
  }

  void ActiveModels::setNavigatorHits(const QVector<int>& activeIndexes)
  {
    int n=_allModels->count(); // Some sort of timing reference (number of model generated so far)
    for(QVector<int>::const_iterator it=activeIndexes.begin(); it!=activeIndexes.end(); it++) {
      at(*it).setNavigatorHitTime(n);
    }
  }

} // namespace DinverCore
