/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-14
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include "SPACParameters.h"
#include "RingPairs.h"

namespace ArrayCore {

/*!
  \class SPACParameters SPACParameters.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

SPACParameters::SPACParameters()
  : ArrayParameters()
{
  windowing().setPeriodCount(100.0);
  setOversamplingFactor(1.0);
  windowing().setSeismicEventTrigger(false);

  _maximumImaginary=1.0;
}

SPACParameters::SPACParameters(const SPACParameters& o)
  : ArrayParameters(o)
{
  _rings=o._rings;
  _maximumImaginary=o._maximumImaginary;
}

void SPACParameters::addRing(double minRadius, double maxRadius)
{
  _rings.append(AutocorrRing(minRadius, maxRadius));
}

int SPACParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
{
  return 6+ArrayParameters::keywordCount();
}

void SPACParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
{
  TRACE;
  int baseIndex=ArrayParameters::keywordCount();
  ArrayParameters::collectKeywords(keywords, prefix, suffix);
  keywords.add(prefix+"OUTPUT TYPES"+suffix, this, baseIndex);         // For compatibility
  keywords.add(prefix+"OUTPUT TARGET FILE"+suffix, this, baseIndex+1); // For compatibility
  keywords.add(prefix+"OUTPUT STMAP FILE"+suffix, this, baseIndex+2);  // For compatibility
  keywords.add(prefix+"OUTPUT MAX FILE"+suffix, this, baseIndex+3);    // For compatibility
  keywords.add(prefix+"RINGS"+suffix, this, baseIndex+4);
  keywords.add(prefix+"MAXIMUM_IMAGINARY_PART"+suffix, this, baseIndex+5);
}

QString SPACParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
{
  TRACE;
  //const SPACParameters * param=parameters();
  // RING_MAXIMUM_ANGLE_STEP=40
  // RING_MAXIMUM_RELATIVE_THICKNESS=0.2
  // RING_MINIMUM_PAIR_COUNT=6
  QString log;
  log+=ArrayParameters::toString(prefix, suffix);
  log+="# List of rings (distances in m): min_radius_1 max_radius_1 min_radius_2 max_radius_2 ...\n";
  log+=prefix+"RINGS"+suffix+"=";
  for(QList<AutocorrRing>::const_iterator it=_rings.begin(); it!=_rings.end(); it++) {
    log+=it->toString()+" ";
  }
  log+="\n";
  log+="# Reject autocorr estimates with an imaginary part above this maximum\n";
  log+=prefix+"MAXIMUM_IMAGINARY_PART"+suffix+"="+QString::number(_maximumImaginary)+"\n";
  return log;
}

bool SPACParameters::setValue(PARAMETERS_SETVALUE_ARGS)
{
  TRACE;
  bool ok=true;
  switch(index-ArrayParameters::keywordCount()) {
  case 0:
    obsoleteKeyword(keywords, 0);
    return true;
  case 1:                      // For compatibility
    obsoleteKeyword(keywords, 1);
    return true;
  case 2:                      // For compatibility
    obsoleteKeyword(keywords, 2);
    return true;
  case 3:                      // For compatibility
    obsoleteKeyword(keywords, 3);
    return true;
  case 4: {
      AutocorrRing r;
      LineParser p(value);
      int index=0;
      _rings.clear();
      while(r.fromString(p, index)) {
        _rings.append(r);
        index+=2;
      }
      return 2*_rings.count()==p.count();
    }
  case 5:
    _maximumImaginary=value.toDouble(&ok);
    return ok;
  default:
    break;
  }
  return ArrayParameters::setValue(index, value, unit, keywords);
}

} // namespace ArrayCore
