/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-08
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FKPARAMETERS_H
#define FKPARAMETERS_H

#include "ArrayParameters.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT FKParameters: public ArrayParameters
  {
  public:
    FKParameters();
    FKParameters(const FKParameters& o);

    enum ProcessType {DirectSteering, RTBF, RTBFRadial, Omni, PoggiVertical, PoggiRadial, Conventional,
                      DirectSteeringRadial, DirectSteeringRefined, DirectSteeringVertical,
                      ActiveRTBF, ActiveDirectSteering, ActiveConventional};

    virtual AbstractParameters * clone() const {return new FKParameters(*this);}

    enum InversionMethod{RefinedGrid, Gradient};

    InversionMethod inversionMethod() const {return _inversionMethod;}
    QString inversionMethodString() const;
    void setInversionMethod(InversionMethod m) {_inversionMethod=m;}
    bool setInversionMethod(QString m);

    void setCacheGridStep(double s) {_cacheGridStep=s;}
    double cacheGridStep() const;

    void setGridStep(double s) {_gridStep=s;}
    double gridStep() const;

    void setGridSize(double s) {_gridSize=s;}
    static double gridSize(double kmax);
    double gridSize() const {return _gridSize;}
    double effectiveGridSize() const;
    double effectiveGridSize(double frequency) const;

    double kmin() const {return _kmin;}
    void setKmin(double k);

    double kmax() const {return _kmax;}
    void setKmax(double k);

    double minimumSlowness() const {return _smin;}
    void setMinimumSlowness(double s) {_smin=s;}

    double maximumSlowness() const {return _smax;}
    void setMaximumSlowness(double s) {_smax=s;}

    double minimumAzimuth() const {return _minAzimuth;}
    void setMinimumAzimuth(double a) {_minAzimuth=a;}

    double maximumAzimuth() const {return _maxAzimuth;}
    void setMaximumAzimuth(double a) {_maxAzimuth=a;}

    void clearCurveRefine();
    void setRefineGridIndex(int gridIndex);
    void addCurveRefine(double f, double minSlow, double maxSlow);

    int refineGridIndex() const {return _refineGridIndex;}
    const Curve<Point2D>& refineMinimumSlowness() const {return _refineSlowMin;}
    const Curve<Point2D>& refineMaximumSlowness() const {return _refineSlowMax;}

    int maximumPeakCount() const {return _maxPeakCount;}
    void setMaximumPeakCount(int m) {_maxPeakCount=m;}

    double absoluteThreshold() const {return _absoluteThreshold;}
    void setAbsoluteThreshold(double a) {_absoluteThreshold=a;}

    double relativeThreshold() const {return _relativeThreshold;}
    void setRelativeThreshold(double r) {_relativeThreshold=r;}

    bool exportAllFKGrids() const {return _exportAllFKGrids;}
    void setExportAllFKGrids(bool g) {_exportAllFKGrids=g;}

    int rotateStepCount() const {return _rotateStepCount;}
    void setRotateStepCount(int c) {_rotateStepCount=c;}

    const QList<QStringList>& arrays() const {return _arrays;}

    ProcessType processType() const {return _processType;}
    bool isActiveProcessType() const;
    QString processTypeString() const;
    void setProcessType(ProcessType p) {_processType=p;}
    bool setProcessType(QString p);

    double damping() const {return _damping>0.0 ? _damping : 0.0;}
    void setDamping(double d) {_damping=d;}

    void setSkipLove(bool s) {_skipLove=s;}
    bool isSkipLove() const {return _skipLove;}

    const QString& fixedEllipticityFileName() const {return _fixedEllipticityFileName;}
    void setFixeElliptiticityFileName(const QString& f) {_fixedEllipticityFileName=f;}

    double minimumDistance() const {return _minimumDistance;}
    void setMinimumDistance(double m) {_minimumDistance=m;}

    double maximumDistance() const {return _maximumDistance;}
    void setMaximumDistance(double m) {_maximumDistance=m;}

    double sourceGridStep() const {return _sourceGridStep;}
    void setSourceGridStep(double s) {_sourceGridStep=s;}

    double sourceGridSize() const {return _sourceGridSize;}
    void setSourceGridSize(double s) {_sourceGridSize=s;}

    virtual QString toString(PARAMETERS_TOSTRING_ARGS_DECL) const;
    virtual void collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS);

    void setDefaultValues(ProcessType t, InversionMethod invm, ArrayStations::Mode m);
    static double defaultGridStepFactor(ProcessType p, InversionMethod m);
  protected:
    virtual int keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const;
    virtual bool setValue(PARAMETERS_SETVALUE_ARGS);
  private:
    ProcessType _processType;
    InversionMethod _inversionMethod;
    int _rotateStepCount;
    int _refineGridIndex;
    int _maxPeakCount;
    bool _skipLove;
    bool _exportAllFKGrids;
    double _kmin, _kmax;
    double _cacheGridStepFactor, _cacheGridStep;
    double _gridStepFactor, _gridStep, _gridSize, _smin, _smax;
    double _minAzimuth, _maxAzimuth;
    Curve<Point2D> _refineSlowMin, _refineSlowMax;
    double _minimumDistance, _maximumDistance;
    double _absoluteThreshold, _relativeThreshold;
    double _damping;
    double _sourceGridStep, _sourceGridSize;
    QList<QStringList> _arrays;
    QString _fixedEllipticityFileName;
  };

} // namespace ArrayCore

#endif // FKPARAMETERS_H
