/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-01-29
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "ConvFKTransverse.h"

namespace ArrayCore {

  /*!
    \class ConvFKTransverse ConvFKTransverse.h
    \brief Transverse conventional FK power

    Transverse conventional FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the transverse direction.
  */

  void ConvFKTransverse::resetCrossSpectrum()
  {
    _crossSpectrum->resetRotatedLove();
  }

  void ConvFKTransverse::addCrossSpectrum()
  {
    _crossSpectrum->addRotatedTransverse();
  }

  void ConvFKTransverse::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRotatedLove(nBlocks);
  }

  double ConvFKTransverse::value(const Point& k, int index) const
  {
    if(isInsideLimits(k)) {
      FKPower p(&FKSteeringOneComponent::cache(_gridCache, index));
      p.setConventionalValue(_crossSpectrum->rotatedLove(k));
      return p.value();
    } else {
      return -1.0;
    }
  }

  double ConvFKTransverse::value(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKSteeringOneComponent * s=static_cast<FKSteeringOneComponent *>(_steering);
      s->initValue(k);
      FKPower p(s);
      p.setConventionalValue(_crossSpectrum->rotatedLove(k));
      return p.value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore
